{
    This file is part of RA1792Control

    RACAL RA1792 Control program

    Copyright (C) 2013-2022 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/.
}

unit uscan;

{$mode objfpc}

{$UNDEF TESTCHART}

interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs, StdCtrls,
  JLabeledIntegerEdit, JButton, TAGraph, TASeries, lclproc, LazUtilities, StrUtils;

type

  { TFSCAN }

  TFSCAN = class(TForm)
    BLOADCSV: TJButton;
    BSAVESCAN: TJButton;
    BCLOSEW: TJButton;
    BSTARTScaN: TJButton;
    BSTOPSCAN: TJButton;
    BCLEARSCAN: TJButton;
    CBAGC: TComboBox;
    DSAVECSV: TSaveDialog;
    LBAGC: TLabel;
    DOPENCSV: TOpenDialog;
    ScanPlot: TChart;
    CBBANDW: TComboBox;
    CBMODE: TComboBox;
    LBBANDW: TLabel;
    LBMODE: TLabel;
    ScanSignalLevel: TLineSeries;
    IEFSTART: TJLabeledIntegerEdit;
    IEFSTOP: TJLabeledIntegerEdit;
    IEFSTEP: TJLabeledIntegerEdit;
    procedure BLOADCSVClick(Sender: TObject);
    procedure BSAVESCANClick(Sender: TObject);
    procedure BCLEARSCANClick(Sender: TObject);
    procedure BCLOSEWClick(Sender: TObject);
    procedure BSTARTScaNClick(Sender: TObject);
    procedure BSTOPSCANClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    function SetupRX: boolean;
    function SetupChart: boolean;
  private

  public

  end;

var
  FSCAN: TFSCAN;
  StopScan: boolean = FALSE;
  SaveSMeterTimer: boolean;

// Declared here for i18n
resourcestring
  BANDSCANFILECORRUPT = 'Band scan file "%s" has invalid data, loading aborted.';
  CANTCREATEFILE = 'Can''t create file "%s".';
  CANTOPENFILE = 'Can''t open file "%s".';
  CANTWRITEFILE = 'Can''t write file "%s".';
  CANTREADFILE = 'Can''t read file "%s".';
  INVALIDSTARTFREQ = 'Invalid start frequency';
  INVALIDSTOPFREQ = 'Invalid stop frequency';
  INVALIDSTEPFREQ = 'Invalid frequency step';
  STARTSCAN = 'Start scan';
  SOMECOMMANDSFAILED = 'One or more commands failed.';
  SCANNING = 'Scanning%s(%d/%d)';
  STOPGREATERSTART = 'Stop frequency must be greater that start frequency.';
  SCANCSVFILENAME = 'Scan-%s-%d-%d.csv';
  SETUPNOTFOUND = 'SETUP line not found, not loading file.';

implementation

{$R *.lfm}

uses ura1792;

{ TFSCAN }

function TFSCAN.SetupRX: boolean;

var tunestep: integer;
    df,sf: real;
    Ok: boolean = TRUE;
    s: string;

begin
    // Check valid scan data
    if IEFSTART.Value > MAXRXF then begin
      ShowMessage(INVALIDSTARTFREQ);
      exit(FALSE);
    end;
    if IEFSTOP.Value > MAXRXF then begin
      ShowMessage(INVALIDSTOPFREQ);
      exit(FALSE);
    end;
    if IEFSTEP.Value > MAXRXF then begin
      ShowMessage(INVALIDSTEPFREQ);
      exit(FALSE);
    end;

    // Step value set not required

    // Set mode
    In_Command := TRUE;
    case CBMODE.ItemIndex of
      0: Ok := Ok and RA1792.SendCommand(Set_Mode, IntToStr(M_AM));
      1: Ok := Ok and RA1792.SendCommand(Set_Mode, IntToStr(M_CW));
      2: Ok := Ok and RA1792.SendCommand(Set_Mode, IntToStr(M_LSB));
      3: Ok := Ok and RA1792.SendCommand(Set_Mode, IntToStr(M_USB));
      4: Ok := Ok and RA1792.SendCommand(Set_Mode, IntToStr(M_FM));
      5: Ok := Ok and RA1792.SendCommand(Set_Mode, IntToStr(M_ISB));
    end;

    // Set bandwidth
    case CBMode.ItemIndex of
      0: Ok := Ok and RA1792.SendCommand(Set_Bandwidth, IntToStr(FILTER_300));
      1: Ok := Ok and RA1792.SendCommand(Set_Bandwidth, IntToStr(FILTER_1000));
      2: Ok := Ok and RA1792.SendCommand(Set_Bandwidth, IntToStr(FILTER_3200));
      3: Ok := Ok and RA1792.SendCommand(Set_Bandwidth, IntToStr(FILTER_6000));
      4: Ok := Ok and RA1792.SendCommand(Set_Bandwidth, IntToStr(FILTER_16000));
    end;

    // Set AGC
    case CBAGC.ItemIndex of
      0: Ok := Ok and RA1792.SendCommand(Set_AGC, IntToStr(AGC_Short));
      1: Ok := Ok and RA1792.SendCommand(Set_AGC, IntToStr(AGC_Medium));
      2: Ok := Ok and RA1792.SendCommand(Set_AGC, IntToStr(AGC_Long));
    end;

    // Set start frequency
    sf := RXState.Freq_rx / 1e6;
    str(sf:9:6,s);
    Ok := Ok and RA1792.SendCommand(Set_Frequency, s);
    In_Command := FALSE;

    // Check if setup succeeded
    if not Ok then
      ShowMessage(SOMECOMMANDSFAILED)
    else begin
      // All ok, so save status timer status and disable it
      SaveSMeterTimer := RA1792.TSMeter.Enabled;
      RA1792.TSMeter.Enabled := FALSE;
      Enable_Status := TRUE;
    end;
    SetupRX := Ok;
end;

// Check bounds and setup chart X axis for selected start/stop frequencies
function TFSCAN.SetupChart: boolean;

begin
  if IEFSTOP.Value <= IEFSTART.Value then begin
    ShowMessage(STOPGREATERSTART);
    Exit(FALSE);
  end else
    SetupChart := TRUE;

  // Setup chart
  with ScanPlot do begin
    AxisList[1].Range.Max := Double(IEFSTOP.Value)/1e6;
    AxisList[1].Range.Min := Double(IEFSTART.Value)/1e6;
    ScanPlot.Bottomaxis.Range.Usemin := TRUE;
    ScanPlot.BottomAxis.Range.Usemax := TRUE;
  end;
end;

// Start scan button
procedure TFSCAN.BSTARTSCANClick(Sender: TObject);

var level, freq, fstep, fstop: double;
    s: string;
    {$IFDEF TESTCHART}
    x, y: double;
    {$ENDIF}
    nsamples: integer;
    Ok: boolean = TRUE;

begin
  if not SetupChart then exit;
  if not SetupRX then exit;
  StopScan := FALSE;
  freq := Double(IEFStart.Value)/1e6;
  fstop := Double(IEFSTOP.Value)/1e6;
  fstep := Double(IEFSTEP.Value)/1e6;
  nsamples := (IEFSTOP.Value - IEFSTART.Value) div IEFSTEP.Value;
  {$IFDEF TESTCHART}
  y:=0;
  randomize;
  repeat
    BSTARTSCAN.LCaption.Caption := Format(SCANNING,
      [LineEnding,ScanSignalLevel.Count,nsamples]);
    x := Random(10000);
    if x >= 9900 then x := Random(100000);
    y := 0.3*(-y+x/100);
    level :=  Double(Trunc(abs(y)));
    if level > 120 then level := 120;
    ScanSignalLevel.AddXY(freq,level);
    freq := freq + fstep;
    Application.ProcessMessages;
  until (freq >= fstop) or StopScan or Error;
  {$ELSE}
  // Do scan
  repeat
    BSTARTSCAN.LCaption.Caption := Format(SCANNING,
      [LineEnding,ScanSignalLevel.Count,nsamples]);

    // read level
    RA1792.GetLevel;
    level := SignalLevel;
    {$IFDEF DEBUGLEVEL}
    RA1792.Message('Level = '+ IntToStr(ScanSignalLevel));
    {$ENDIF}
    ScanSignalLevel.AddXY(freq,level);

    // tune to next frequency
    freq := freq + fstep;
    str(freq:9:6,s);
    In_Command := TRUE;
    Ok := Ok and RA1792.SendCommand(Set_Frequency, s);
    In_Command := FALSE;
    Application.ProcessMessages;
  until (freq >= fstop) or StopScan or not Ok;
  {$ENDIF}
  BSTARTSCAN.LCaption.Caption := STARTSCAN;
end;

// Stop scan button
procedure TFSCAN.BSTOPSCANClick(Sender: TObject);
begin
  StopScan := TRUE;
  BSTARTSCAN.LCaption.Caption := STARTSCAN;
end;

procedure TFSCAN.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
    BCLOSEWClick(Self);
    CloseAction := caNone;
end;

// Initializations at startup
procedure TFSCAN.FormCreate(Sender: TObject);
begin
  SaveSMeterTimer := RA1792.TSmeter.Enabled;
  RA1792.TSmeter.Enabled := FALSE;
end;

procedure TFSCAN.FormShow(Sender: TObject);
begin
  // Clear all Edits and combos
  IEFSTART.Clear;
  IEFSTOP.Clear;
  IEFSTEP.Clear;
  CBMode.ItemIndex := -1;
  CBBANDW.ItemIndex := -1;
  CBAGC.ItemIndex := -1;
end;

// Clear scan button
procedure TFSCAN.BCLEARSCANClick(Sender: TObject);
begin
  ScanSignalLevel.Clear;
  ScanPlot.Bottomaxis.Range.Min := 0;
  ScanPlot.BottomAxis.Range.Max := 30;
  ScanPlot.Bottomaxis.Range.Usemin := TRUE;
  ScanPlot.BottomAxis.Range.Usemax := TRUE;
  IEFSTART.Clear;
  IEFSTOP.Clear;
  IEFSTEP.Clear;
  CBMode.ItemIndex := -1;
  CBBANDW.ItemIndex := -1;
  CBAGC.ItemIndex := -1;
end;

// Save scan data
procedure TFSCAN.BSAVESCANClick(Sender: TObject);

var i: integer;
    f: textfile;
    s: string;
    freq,level: Double;

begin
  DSAVECSV.Filename := Format(SCANCSVFILENAME,
                    [FormatDateTime('DD-MM-YYYY-hh-mm-ss',now),IEFSTART.Value,
                    IEFSTOP.Value]);

  DSAVECSV.Initialdir := BandScansDir;
  if DSAVECSV.Execute then begin
    try
      AssignFile(f,DSAVECSV.Filename);
      Rewrite(f);
    except
      ShowMessage(Format(CANTCREATEFILE, [DSAVECSV.Filename]));
    end;
    // write this scan configuration
    WriteLn(f,Format('"SETUP";"%d:%d:%d:%d:%d:%d:"', [IEFSTART.Value, IEFSTOP.Value,
              IEFSTEP.Value,CBMode.ItemIndex,CBBANDW.ItemIndex,CBAGC.ItemIndex]));

    // Save data points
    for i := 0 to ScanSignalLevel.Count - 1 do begin
      freq := ScanSignalLevel.GetXValue(i);
      level := ScanSignalLevel.GetYValue(i);
      s := FormatFloat('00.000000',freq)+';'+FormatFloat('000',level);
      try
        WriteLn(f,s);
      except
        ShowMessage(Format(CANTWRITEFILE, [DSAVECSV.Filename]));
      end;
    end;
    CloseFile(f);
  end;
end;

// Load & display CSV data saved
procedure TFSCAN.BLOADCSVClick(Sender: TObject);

var f: textfile;
    s,st: string;
    p: integer;
    freq, level: Double;

begin
  DOPENCSV.Initialdir := BandScansDir;
  ScanPlot.Bottomaxis.Range.Usemin := FALSE;
  ScanPlot.BottomAxis.Range.Usemax := FALSE;
  if DOPENCSV.Execute then begin
    ScanSignalLevel.Clear;
    try
      AssignFile(f,DOPENCSV.Filename);
      Reset(f);
    except
      ShowMessage(Format(CANTOPENFILE, [ExtractFileName(DOPENCSV.Filename)]));
    end;

    // Read and set up scan configuration
    try
      ReadLn(f,s);
    except
      ShowMessage(Format(CANTREADFILE, [ExtractFileName(DOPENCSV.Filename)]));
    end;
    if Pos('SETUP',s) <> 0 then begin
      p := pos(';',s);
      if p > 0 then begin
        s := copy(s, p+2, Length(s)-1);
      end;

      // Parse configuration fields
      st :=  ExtractDelimited(1,s,[':']);
      IEFSTART.Value := StrToInt(st);

      st := ExtractDelimited(2,s,[':']);
      IEFSTOP.Value := StrToInt(st);

      st := ExtractDelimited(3,s,[':']);
      IEFSTEP.Value := StrToInt(st);

      st := ExtractDelimited(4,s,[':']);
      CBMODE.ItemIndex := StrToInt(st);
      case CBMODE.ItemIndex of
        0,1: CBBANDW.Enabled := TRUE;
        2,3: CBBANDW.Enabled := FALSE;
      end;

      st := ExtractDelimited(5,s,[':']);
      CBBANDW.ItemIndex := StrToInt(st);

      st := ExtractDelimited(6,s,[':','"']);
      CBAGC.ItemIndex := StrToInt(st);
    end else begin
      ShowMessage(SETUPNOTFOUND);
      exit;
    end;

    // Read and show scan data
    while not Eof(f) do begin
      try
        ReadLn(f,s);
      except
        ShowMessage(Format(CANTREADFILE, [DOPENCSV.Filename]));
        exit;
      end;

      st := ExtractDelimited(1,s,[';']);
      try
        freq := LazUtilities.StrToDouble(st);
      except
        On E: exception do begin
          ShowMessage(Format(BANDSCANFILECORRUPT,[ExtractFileName(DOPENCSV.Filename)]));
          ScanSignalLevel.Clear;
          CloseFile(f);
          exit;
        end;
      end;
      st := ExtractDelimited(2,s,[';']);
      try
        level := LazUtilities.StrToDouble(st);
      except
        On E: exception do begin
          ShowMessage(Format(BANDSCANFILECORRUPT,[ExtractFileName(DOPENCSV.Filename)]));
          ScanSignalLevel.Clear;
          CloseFile(f);
          exit;
        end;
      end;
      ScanSignalLevel.AddXY(freq,level);
    end;
    CloseFile(f);
  end;
end;

// Close window button
procedure TFSCAN.BCLOSEWClick(Sender: TObject);

begin
   StopScan := TRUE;
   FSCAN.Hide;
   with RA1792 do begin
      RestoreState;
      TSMeter.Enabled := SaveSMeterTimer;
      Enable_Status := SaveSMeterTimer
   end;
   RA1792.RestoreState;
end;

end.

