{
    This file is part of RA3701Control

    RACAL RA3701 Control program

    Copyright (C) 2023 G. Perotti, I1EPJ, i1epj@aricasale.it

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/.

                                   * * *

    Allows entering the scan parameters and perform a channel scan
}

unit uchscan;

{$mode ObjFPC}{$H+}

interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs, StdCtrls,
  Spin, EditBtn;

type

  { TFCHSCAN }

  TFCHSCAN = class(TForm)
    BAPPLY: TButton;
    BCLOSE: TButton;
    BSTARTSTOPSCAN: TButton;
    CBCOR: TComboBox;
    CBDIR: TComboBox;
    FSPDWELL: TFloatSpinEdit;
    LBCOR: TLabel;
    LBDIR: TLabel;
    LBCORLEVEL: TLabel;
    LBSTART: TLabel;
    LBDWELL: TLabel;
    LBSTOP: TLabel;
    SPCORLEVEL: TSpinEdit;
    SPCHSTART: TSpinEdit;
    SPCHEND: TSpinEdit;
    procedure BSTARTSTOPSCANClick(Sender: TObject);
    procedure BCLOSEClick(Sender: TObject);
    procedure BAPPLYClick(Sender: TObject);
    procedure CBCORChange(Sender: TObject);
    procedure CBDIRChange(Sender: TObject);
    procedure EDDWELLChange(Sender: TObject);
    procedure EDSTARTCHChange(Sender: TObject);
    procedure EDSTOPCHChange(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FSPDWELLChange(Sender: TObject);
    procedure SPCHENDChange(Sender: TObject);
    procedure SPCHSTARTChange(Sender: TObject);
  private

  public

  end;

var
  FCHSCAN: TFCHSCAN;
  iStart,iStop: integer;
  fDwell: real;
  Direction: boolean;

implementation

uses ura3701;

{ TFCHSCAN }

// Things to do when showing form
procedure TFCHSCAN.FormShow(Sender: TObject);
begin
  FCHSCAN.Caption := ENTERSCANPARMS;
  BSTARTSTOPSCAN.Caption := STARTSCAN;
  BAPPLY.Caption := APPLY;
  BCLOSE.Caption := CLOSEW;
  LBSTART.Caption := STARTCH;
  LBSTOP.Caption := STOPCH;
  LBDWELL.Caption := DWELLTIME;
  LBCOR.Caption := CORCONTROL;
  LBDIR.Caption := DIRECT;
  LBCORLEVEL.Caption := CORLEVEL;

  if CBCOR.ItemIndex = 0 then begin
    LBCORLEVEL.Enabled := FALSE;
    SPCORLEVEL.Enabled := FALSE;
  end else begin
    LBCORLEVEL.Enabled := TRUE;
    SPCORLEVEL.Enabled := TRUE;
  end;
  if RA3701.MUSEPROGCH.Checked then begin
    BSTARTSTOPSCAN.Hide;
    CBCOR.Enabled := FALSE;
    LBCORLEVEL.Enabled := FALSE;
    SPCHSTART.Value := 0;
    SPCHEND.Value := 99;
    FSPDWELL.Value := 1.00;
    CBDIR.ItemIndex := 0;
  end else begin
    CBCOR.Enabled := TRUE;
    LBCOR.Enabled := TRUE;
    if CBCOR.ItemIndex = 0 then
      LBCORLEVEL.Enabled := FALSE
    else
      LBCORLEVEL.Enabled := TRUE;
    BSTARTSTOPSCAN.Hide;
  end;
  StartSc := FALSE;
end;

procedure TFCHSCAN.FSPDWELLChange(Sender: TObject);
begin
  BSTARTSTOPSCAN.Hide;
end;


//
procedure TFCHSCAN.SPCHENDChange(Sender: TObject);
begin
  SPCHSTART.Font.Color := clDefault;
  SPCHEND.Font.Color := clDefault;
  BSTARTSTOPSCAN.Hide;
end;

procedure TFCHSCAN.SPCHSTARTChange(Sender: TObject);
begin
  SPCHSTART.Font.Color := clDefault;
  SPCHEND.Font.Color := clDefault;
  BSTARTSTOPSCAN.Hide;
end;

//
// Form buttons
//

// "Apply" button
procedure TFCHSCAN.BAPPLYClick(Sender: TObject);

var par,cor, corl: string;
    Parms_OK: boolean;

begin
  Parms_OK := TRUE;
  iStart := SPCHSTART.Value;
  iStop  := SPCHEND.Value;
  if not (iStop > iStart) then begin
    SPCHSTART.Font.Color := clRED;
    SPCHEND.Font.Color := clRED;
    Parms_OK := FALSE;
  end;
  fDwell := FSPDWELL.Value;
  Direction := (CBDIR.ItemIndex = 0);    // 0 = up, 1 = down
  if RA3701.MUSEPROGCH.Checked then begin
    if Parms_OK then begin
      fDwell := fDwell/86400; // convert seconds to day fraction
      BSTARTSTOPSCAN.Caption := STARTSCAN;
      BSTARTSTOPSCAN.Show;
    end;
  end else begin
    cor := IntToStr(CBCOR.ItemIndex);
    corl := IntToStr(SPCORLEVEL.Value);
    if In_Command then exit;
    In_Command := TRUE;
    // Set COR/Squelch level
    if RA3701.SendCommand(Set_CORSQLevel, corl) then begin
      if not RA3701.GetAck then RA3701.MsgToShow := NOACK;
    end else RA3701.MsgToShow := COMMANDFAILED;
    // Set scan parameters scan
    par := Format('%2.2d,%2.2d,%2.2f,%s',[iStart, iStop, fDwell, cor]);
    if RA3701.SendCommand(Set_ScanParms, par) then begin
      if not RA3701.GetAck then RA3701.MsgToShow := NOACK;
    end else RA3701.MsgToShow := COMMANDFAILED;
    In_Command := FALSE;
    BSTARTSTOPSCAN.Caption := STARTSCAN;
    BSTARTSTOPSCAN.Show;
  end;
end;

// "Start scan" or "Stop scan" button
procedure TFCHSCAN.BSTARTSTOPSCANClick(Sender: TObject);

var Dir,stmp: string;
    i: integer;
    t: TDateTime;
    SomeChanScanned: boolean;

begin
  if RA3701.MUSEPROGCH.Checked then begin
    StartSc := NOT StartSc;
    with RA3701 do begin
      BSTARTSTOPSCAN.Caption := STOPSCAN;
      i := iStart;
      KeyboardState := STATE_SCAN;
      if Startsc then
        BSTARTSTOPSCAN.Caption := STOPSCAN
      else
        BSTARTSTOPSCAN.Caption := STARTSCAN;
      DoMessages;
      while StartSc do begin
        if RXChannels[i].ScanFlag then begin
          SomeChanScanned := TRUE;
          stmp := Format('%2.2d',[i]);
          RA3701.PutChanString(stmp);
          RXState := RXChannels[i];
          RestoreState;
          UpdateDisplay;
          DoMessages;
          t := Time+fDwell;
          while Time < t do DoMessages;
        end;
        if Direction then i := i + 1 else i := i - 1;
        if i > iStop then begin
          if SomeChanScanned then begin
            i := iStart
          end else begin
            MsgToShow := NOCHANSCANF;
          end;
        end;
        if i < iStart then begin
          if SomeChanScanned then begin
            i := iStop
          end else begin
            MsgToShow := NOCHANSCANF;
          end;
        end;
      end;
    end;
  end else begin
    StartSc := NOT StartSc;
    LBCOR.Enabled := TRUE;
    RA3701.LBScan.Visible := StartSc;
    if In_Command then exit;
    In_Command := TRUE;
    Dir := IntToStr(CBDIR.ItemIndex);
    if StartSc then begin
      KeyboardState := STATE_SCAN;
      if RA3701.SendCommand(Set_Scan, '2,'+Dir) then begin
        if not RA3701.GetAck then RA3701.MsgToShow := NOACK;
      end else RA3701.MsgToShow := COMMANDFAILED;
      BSTARTSTOPSCAN.Caption := STOPSCAN;
    end else begin
      if RA3701.SendCommand(Set_Scan, '0,'+Dir) then begin
        if not RA3701.GetAck then RA3701.MsgToShow := NOACK;
      end else RA3701.MsgToShow := COMMANDFAILED;
      BSTARTSTOPSCAN.Caption := STARTSCAN;
    end;
  end;
end;

// "Close" button
procedure TFCHSCAN.BCLOSEClick(Sender: TObject);
begin
  StartSc := FALSE;
  if not RA3701.MUSEPROGCH.Checked then begin
    if RA3701.SendCommand(Set_Scan, '0,'+IntToStr(CBDIR.ItemIndex)) then begin
      if not RA3701.GetAck then RA3701.MsgToShow := NOACK+Status;
    end else RA3701.MsgToShow := COMMANDFAILED;
    BSTARTSTOPSCAN.Caption := STARTSCAN;
  end;
  FCHSCAN.Hide;
  KeyboardState := STATE_NORMAL;
  if RA3701.MRESTSTATE.Checked then begin
    // Restore old state after scan.
    // If not checked, RX stays at frequency&mode of the last channel scanned
    RXState := OldRXState;
    RA3701.RestoreState;
  end;
  RA3701.GetStatus;
  RA3701.ParseStatus;
  RA3701.SetActiveButtons;
  RA3701.UpdateDisplay;
end;

//
// Handles modifications to scan values
//

// COR
procedure TFCHSCAN.CBCORChange(Sender: TObject);
begin
  if CBCOR.ItemIndex = 0 then begin
    LBCORLEVEL.Enabled := FALSE;
    SPCORLEVEL.Enabled := FALSE;
  end else begin
    LBCORLEVEL.Enabled := TRUE;
    SPCORLEVEL.Enabled := TRUE;
  end;
  BSTARTSTOPSCAN.Hide;
end;

// Direction
procedure TFCHSCAN.CBDIRChange(Sender: TObject);
begin
  BSTARTSTOPSCAN.Hide;
end;

procedure TFCHSCAN.EDDWELLChange(Sender: TObject);
begin
  BSTARTSTOPSCAN.Hide;
end;

procedure TFCHSCAN.EDSTARTCHChange(Sender: TObject);
begin
  BSTARTSTOPSCAN.Hide;
end;

procedure TFCHSCAN.EDSTOPCHChange(Sender: TObject);
begin
  BSTARTSTOPSCAN.Hide;
end;


initialization
  {$I uchscan.lrs}

end.

